(function () {
  let ws;
  const serverIp = window.location.hostname;
  const serverPort = 8085;
  let lockOpacity = 300;
  const roomId = localStorage.getItem("ROOM_ID");
  const clientId = localStorage.getItem("CLIENT_ID");

  if (!roomId) {
    alert("⚠️ Không tìm thấy Room ID. Vui lòng quay lại kết nối!");
    window.location.href = "/index.html";
    return;
  }

  let players = [];
  let gameStart = false;
  let gameRunning = false;

  let currentIndex = players.findIndex((p) => p.isActive) || 0;
  let isRunning = false;
  let timeMode = 0;
  // let isRunning = JSON.parse(localStorage.getItem("IS_PAUSED")) || false;

  const titleEl = document.querySelector(".title");
  const btnDisconnect = document.getElementById("btnDisconnect");
  const btnTakeTurn = document.getElementById("take-turns");
  const btnAskForTime = document.getElementById("ask-for-time");
  const btnIncrease = document.getElementById("increase");
  const btnPause = document.getElementById("btn-pause");
  const btnDesIncrease = document.getElementById("disIncrease");

  // Kết nối ban đầu
  connectWebSocket();

  function renderPlayer() {
    if (players.length > 0) {
      const currentPlayer = players[currentIndex];
      titleEl.textContent = currentPlayer?.name || "Chưa có người chơi";
      // localStorage.setItem("PLAYERS", JSON.stringify(players));
    }
  }

  function updatePauseButtonVisibility() {
    const btnPause = document.getElementById("btn-pause");
    if (!btnPause) return;

    if (timeMode === 1) {
      btnPause.classList.remove("btn-hidden");
      btnAskForTime.classList.remove("btn-hidden");
    } else {
      btnPause.classList.add("btn-hidden");
      btnAskForTime.classList.add("btn-hidden");
    }
  }

  function updatePauseButton() {
    const btnPause = document.getElementById("btn-pause");
    if (!btnPause) return;

    // Ẩn/hiện nút dựa trên timeMode
    if (timeMode === 1) {
      btnPause.classList.remove("btn-hidden");

      // Cập nhật trạng thái nút
      btnPause.textContent = !isRunning ? "TIẾP TỤC GIỜ" : "TẠM DỪNG GIỜ";
      btnPause.style.backgroundColor = !isRunning ? "green" : "red";
      btnPause.style.color = "white";
    } else {
      btnPause.classList.add("btn-hidden");
    }

    // Lưu trạng thái vào localStorage
    localStorage.setItem("IS_PAUSED", JSON.stringify(isRunning));
  }

  function sendData(activeIndex, action) {
    if (ws && ws.readyState === WebSocket.OPEN) {
      let payload = null;

      if (action === "switch_turn") {
        payload = {
          type: "command",
          action: "switch_turn",
          data: {
            playerId: players[activeIndex].id,
            isActive: players[activeIndex].isActive,
          },
        };
      }
      if (action === "update_score") {
        payload = {
          type: "command",
          action: "update_score",
          data: {
            playerId: players[activeIndex].id,
            name: players[activeIndex].name,
            score: players[activeIndex].score,
            isActive: players[activeIndex].isActive,
          },
        };
      }
      if (action === "request_extra_time") {
        payload = {
          type: "command",
          action: "request_extra_time",
          data: {
            playerId: players[activeIndex].id,
            extraTime: players[activeIndex].extraTime,
            isActive: players[activeIndex].isActive,
          },
        };
      }
      if (action === "toggle_pause") {
        payload = {
          type: "command",
          action: "toggle_pause",
          data: {
            isRunning: isRunning,
          },
        };
      }
      if (payload) {
        ws.send(JSON.stringify(payload));
      }
    }
  }

  function connectWebSocket() {
    ws = new WebSocket(SOCKET_URL);

    ws.onopen = () => {
      // Join room lại
      ws.send(
        JSON.stringify({
          type: "join-room",
          room: roomId,
          clientId: clientId,
          device: "web",
        }),
      );

      ws.send(
        JSON.stringify({
          type: "get-initial-data",
          from: clientId, // để biết gửi lại cho ai
          room: roomId,
        }),
      );
    };

    ws.onmessage = (event) => {
      const message = JSON.parse(event.data);

      switch (message.type) {
        case "initial-data":
          players = message.data.players || [];
          gameStart = message.data.gameStart || false;
          gameRunning = message.data.gameRunning || false;
          isRunning = message.data.isRunning;
          matchType = message.data.matchType;
          timeMode = message.data.timeMode;
          console.log("message", message.data);
          const activeIdx = players.findIndex((p) => p.isActive);
          currentIndex = activeIdx !== -1 ? activeIdx : 0;
          console.log("message", message.data);

          updatePauseButtonVisibility();
          updatePauseButton();
          renderPlayer();
          break;

        case "control":
          switch (message.action) {
            case "request_extra_time":
              const playerExtraTime = players.find(
                (p) => p.id === message.data.playerId,
              );
              if (playerExtraTime) {
                playerExtraTime.extraTime = message.data.extraTime;
                renderPlayer();

                if (playerExtraTime.isActive) {
                  const countdownDuration = 10 * 1000; // 10s
                  const endTime = Date.now() + countdownDuration;

                  // lưu vào localStorage
                  localStorage.setItem("extraTimeEnd", endTime.toString());

                  startCountdown(endTime);
                }
              }
              break;

            case "toggle_pause":
              isRunning = message.data.isRunning;
              updatePauseButton();
              break;
            case "app_toggle_pause":
              isRunning = message.data.isRunning;
              updatePauseButton();
              break;

            case "change_name":
              const playerChangeName = players.find(
                (p) => p.id === message.data.playerId,
              );
              if (playerChangeName) playerChangeName.name = message.data.name;

              renderPlayer();
              break;

            case "app_change_name":
              const { playerId, name } = message.data;

              const playerToRename = players.find((p) => p.id === playerId);
              if (playerToRename) playerToRename.name = name;

              renderPlayer();
              break;

            case "update_score":
              const { playerId: id } = message.data;
              const player = players.find((p) => p.id === id);
              if (player) player.score == message.data.score;
              renderPlayer();
              break;

            case "app_update_score":
              const { playerId: pid, score } = message.data;
              const playerToUpdate = players.find((p) => p.id === pid);
              if (playerToUpdate) playerToUpdate.score += score;
              renderPlayer();
              break;
            case "app_update_score_action":
              stopCountdown();
              break;
            case "app_update_game_start":
              gameStart = message.data.gameStart;
              renderPlayer();
              break;

            case "app_update_game_running":
              gameRunning = message.data.gameRunning;
              renderPlayer();
              break;

            case "app_switch_turn":
              const currentIdx = players.findIndex(
                (p) => p.id === message.data.playerId,
              );

              if (currentIdx !== -1) {
                const nextIndex = (currentIdx + 1) % players.length;

                players = players.map((p, i) => ({
                  ...p,
                  isActive: i === nextIndex, // chỉ thằng kế tiếp là true
                }));
                const activeIdx = players.findIndex((p) => p.isActive);
                currentIndex = activeIdx !== -1 ? activeIdx : 0;
              }

              stopCountdown();
              renderPlayer();
              break;

            case "switch_turn":
              const activeId = message.data.playerId;

              players = players.map((p) => ({
                ...p,
                isActive: p.id === activeId, // chỉ playerId này được true
              }));

              stopCountdown();
              renderPlayer();
              break;

            case "disconnect":
              const { room, playerIds, device } = message.data;
              // Xóa room khỏi object rooms
              if (room) {
                delete room;
                window.location.href = "/index.html";
              }

              break;
          }
          break;

        default:
          console.log("Unknown message:", message);
      }
    };

    ws.onclose = () => console.log("🔌 WebSocket ngắt kết nối");
    ws.onerror = (e) => console.error("❌ Lỗi WebSocket:", e);
  }

  // --- Nút ngắt kết nối ---
  btnDisconnect.addEventListener("click", () => {
    if (ws && ws.readyState === WebSocket.OPEN) {
      ws.send(JSON.stringify({ type: "disconnect", room: roomId }));
    }
    // localStorage.clear();
    sessionStorage.clear();
    window.location.href = "/index.html";
  });

  // --- Nút đổi lượt ---
  // --- Nút đổi lượt ---
  btnTakeTurn.addEventListener("click", () => {
    if (!gameStart || !gameRunning || (!isRunning && timeMode === 1)) return;
    if (players.length === 0) return;

    // ✅ Thêm chống spam
    btnTakeTurn.disabled = true;
    btnTakeTurn.style.opacity = "0.5";

    players[currentIndex].isActive = false;
    currentIndex = (currentIndex + 1) % players.length;
    players[currentIndex].isActive = true;
    localStorage.setItem("CURRENT_NAME_PLAYER", players[currentIndex].name);
    renderPlayer();
    sendData(currentIndex, "switch_turn");
    stopCountdown();

    btnAskForTime.textContent = "XIN THỜI GIAN";
    canAskForTime = true;
    if (countdownInterval) {
      clearInterval(countdownInterval);
      countdownInterval = null;
    }

    // ✅ Kích hoạt lại sau 300ms
    setTimeout(() => {
      btnTakeTurn.disabled = false;
      btnTakeTurn.style.opacity = "1";
    }, lockOpacity);
  });

  // --- Nút xin thời gian ---
  let canAskForTime = true;
  let countdownInterval = null;
  btnAskForTime.addEventListener("click", () => {
    if (!gameStart || !gameRunning || (!isRunning && timeMode === 1)) return;
    if (!canAskForTime) return;

    const currentPlayer = players[currentIndex];
    if (!currentPlayer?.isActive || currentPlayer.extraTime <= 0) return;
    if (gameStart && gameRunning) {
      currentPlayer.extraTime -= 1;
      renderPlayer();
      sendData(currentIndex, "request_extra_time");

      // Bắt đầu cooldown 10s
      canAskForTime = false;
      let countdown = 10;
      const originalText = btnAskForTime.textContent;

      // Hiển thị ngay 10s
      btnAskForTime.textContent = `${countdown}s`;

      countdownInterval = setInterval(() => {
        countdown--;
        btnAskForTime.textContent = `${countdown}s`;

        if (countdown <= 0) {
          clearInterval(countdownInterval);
          btnAskForTime.textContent = originalText;
          canAskForTime = true;
        }
      }, 1000);
    }
  });
  // --- Nút trừ điểm ---
  btnDesIncrease.addEventListener("click", () => {
    if (!gameStart || !gameRunning || (!isRunning && timeMode === 1)) return;
    btnDesIncrease.disabled = true;
    btnDesIncrease.style.opacity = "0.5";
    const currentPlayer = players[currentIndex];
    if (!currentPlayer?.isActive) return;
    currentPlayer.score -= 1;
    renderPlayer();
    sendData(currentIndex, "update_score");
    setTimeout(() => {
      btnDesIncrease.disabled = false;
      btnDesIncrease.style.opacity = "1";
    }, lockOpacity);
  });
  // --- Nút tăng điểm ---
  btnIncrease.addEventListener("click", () => {
    if (!gameStart || !gameRunning || (!isRunning && timeMode === 1)) return;
    btnIncrease.disabled = true;
    btnIncrease.style.opacity = "0.5";
    const currentPlayer = players[currentIndex];
    if (!currentPlayer?.isActive) return;
    currentPlayer.score += 1;
    renderPlayer();
    sendData(currentIndex, "update_score");
    setTimeout(() => {
      btnIncrease.disabled = false;
      btnIncrease.style.opacity = "1";
    }, lockOpacity);
  });

  // --- Nút tạm dừng ---
  btnPause.addEventListener("click", () => {
    if (!gameStart || !gameRunning) return;
    if (timeMode !== 1) return; // Chỉ hoạt động khi timeMode = 1

    isRunning = !isRunning;
    updatePauseButton();
    sendData(currentIndex, "toggle_pause");
  });

  // ==== hàm đếm thời gian 10s khi load lai trang vẫn tiếp tục đếm ===
  function startCountdown(endTime) {
    if (countdownInterval) clearInterval(countdownInterval);
    btnAskForTime.disabled = true;

    countdownInterval = setInterval(() => {
      const remaining = Math.max(0, Math.ceil((endTime - Date.now()) / 1000));
      btnAskForTime.textContent = `${remaining}s`;

      if (remaining <= 0) {
        clearInterval(countdownInterval);
        localStorage.removeItem("extraTimeEnd");
        btnAskForTime.disabled = false;
        btnAskForTime.textContent = "XIN THỜI GIAN";
      }
    }, 200);
  }

  function stopCountdown() {
    // Dừng interval
    if (countdownInterval) {
      clearInterval(countdownInterval);
      countdownInterval = null;
    }

    // Reset UI
    btnAskForTime.disabled = false;
    btnAskForTime.textContent = "XIN THỜI GIAN";

    // Xóa dữ liệu localStorage
    localStorage.removeItem("extraTimeEnd");

    // Reset biến trạng thái (nếu có)
    canAskForTime = true;
  }

  window.addEventListener("load", () => {
    const savedEnd = localStorage.getItem("extraTimeEnd");
    if (savedEnd) {
      const endTime = Number(savedEnd);
      if (Date.now() < endTime) {
        startCountdown(endTime); // tiếp tục countdown
      } else {
        localStorage.removeItem("extraTimeEnd");
      }
    }
  });

  // Khởi chạy
  // updatePauseButtonVisibility();
  updatePauseButton();
  renderPlayer();
  connectWebSocket();
})();
