(function () {
  let heartbeatInterval = null;
  let pingTimeout = null;
  let isAlive = false;
  let players = [];
  let isRunning = false;
  let activeIndex = -1;
  let totalRounds = 1;
  let gameStart = false;
  let gameRunning = false;
  let lockOpacity = 500;
  let timeOut = 0;
  let timeMode = 0;
  let ws;
  const roomId = localStorage.getItem("ROOM_ID");
  const clientId = localStorage.getItem("CLIENT_ID");
  const colors = ["#ffffff", "#FFCC00", "#3399FF", "#99CC00", "#FFCC99"];
  const btnDecreaseScore = document.getElementById("decreaseScore");
  const btnIncreaseScore = document.getElementById("increaseScore");
  const btnMoreTime = document.getElementById("moreTime");
  const btnNextTurn = document.getElementById("nextTurn");

  function updatePauseButtonVisibility() {
    const btnPause = document.getElementById("btnPause");
    if (!btnPause) return;

    if (timeMode === 1) {
      btnPause.classList.remove("btn-hidden");
      btnMoreTime.classList.remove("btn-hidden");
    } else {
      btnPause.classList.add("btn-hidden");
      btnMoreTime.classList.add("btn-hidden");
    }
  }

  function stopHeartbeat() {
    if (heartbeatInterval) {
      clearInterval(heartbeatInterval);
      heartbeatInterval = null;
    }
    if (pingTimeout) {
      clearTimeout(pingTimeout);
      pingTimeout = null;
    }
    isAlive = false;
  }

  function startHeartbeat() {
    stopHeartbeat();

    isAlive = true;

    heartbeatInterval = setInterval(() => {
      if (ws && ws.readyState === WebSocket.OPEN) {
        if (!isAlive) {
          console.warn("No pong received, closing connection");
          ws.close();
          return;
        }

        isAlive = false;

        ws.send(
          JSON.stringify({
            type: "ping",
            timestamp: Date.now(),
          }),
        );

        pingTimeout = setTimeout(() => {
          if (!isAlive) {
            console.warn("Ping timeout, closing connection");
            ws.close();
          }
        }, 10000);
      }
    }, 30000);
  }

  function connectWebSocket() {
    try {
      ws = new WebSocket(SOCKET_URL);
      setupWebSocketEvents();
    } catch (error) {
      console.error("WebSocket connection error:", error);
    }
  }

  function setupWebSocketEvents() {
    if (!ws) return;
    ws.onopen = () => {
      ws.send(
        JSON.stringify({
          type: "join-room",
          room: roomId,
          clientId: clientId,
          device: "web",
        }),
      );

      ws.send(
        JSON.stringify({
          type: "get-initial-data",
          from: clientId,
          room: roomId,
        }),
      );
      startHeartbeat();
    };

    ws.onmessage = (event) => {
      try {
        const message = JSON.parse(event.data);

        if (message.type === "pong") {
          isAlive = true;
          if (pingTimeout) {
            clearTimeout(pingTimeout);
            pingTimeout = null;
          }
          return;
        }

        if (message.type === "ping") {
          ws.send(
            JSON.stringify({
              type: "pong",
              timestamp: message.timestamp,
            }),
          );
          return;
        }

        switch (message.type) {
          case "initial-data":
            players = message.data.players || [];
            isRunning = message.data.isRunning;
            totalRounds = message.data.totalRounds;
            gameStart = message.data.gameStart || false;
            gameRunning = message.data.gameRunning || false;
            timeOut = message.data.timeOut;
            timeMode = message.data.timeMode;
            console.log("message", message.data);
            updatePauseButtonVisibility();
            renderPlayers();
            updatePauseButton();
            break;

          case "control":
            switch (message.action) {
              case "request_extra_time":
                const playerExtraTime = players.find(
                  (p) => p.id === message.data.playerId,
                );
                if (playerExtraTime) {
                  playerExtraTime.extraTime = message.data.extraTime;
                  updateUI();

                  if (playerExtraTime.isActive) {
                    const countdownDuration = 10 * 1000;
                    const endTime = Date.now() + countdownDuration;
                    localStorage.setItem("extraTimeEnd", endTime.toString());
                    startCountdown(endTime);
                  }
                }
                break;

              case "app_update_players":
                const data = message.data;
                const target = players.find((p) => p.id === data.playerId);
                if (target) {
                  target.hr = data.hr;
                  target.avg = Number(data.avg ?? 0).toFixed(3);
                }
                renderPlayers();
                break;
              case "app_update_total_round":
                totalRounds = message.data.totalRounds;
                renderPlayers();
                break;
              case "app_update_game_start":
                gameStart = message.data.gameStart;
                gameRunning = true;
                renderPlayers();
                break;
              case "app_update_game_running":
                gameRunning = message.data.gameRunning;
                renderPlayers();
                break;
              case "toggle_pause":
                isRunning = message.data.isRunning;
                updatePauseButton();
                renderPlayers();
                break;

              case "change_name":
                const { playerId, name } = message.data;
                const playerToRename = players.find((p) => p.id === playerId);
                if (playerToRename) playerToRename.name = name;
                renderPlayers();
                break;

              case "update_score":
                const { playerId: id } = message.data;
                const player = players.find((p) => p.id === id);
                if (player) player.score = message.data.score;
                renderPlayers();
                break;

              case "app_update_score":
                const { playerId: pid, score } = message.data;
                const playerToUpdate = players.find((p) => p.id === pid);
                if (playerToUpdate) playerToUpdate.score += score;
                renderPlayers();
                break;

              case "app_switch_turn":
                const currentIndex = players.findIndex(
                  (p) => p.id === message.data.playerId,
                );

                if (currentIndex !== -1) {
                  const nextIndex = (currentIndex + 1) % players.length;
                  players = players.map((p, i) => ({
                    ...p,
                    isActive: i === nextIndex,
                  }));
                }
                stopCountdown();
                renderPlayers();
                break;
              case "app_update_score_action":
                stopCountdown();
                break;
              case "switch_turn":
                const activeId = message.data.playerId;
                players = players.map((p) => ({
                  ...p,
                  isActive: p.id === activeId,
                }));
                renderPlayers();
                break;

              case "disconnect":
                const { room, playerIds, device } = message.data;
                if (room) {
                  window.location.href = "/index.html";
                }
                break;
            }
            break;

          default:
            console.log("Unknown message:", message);
        }
      } catch (e) {
        console.error("Error parsing message:", e);
      }
    };

    ws.onclose = (event) => {
      console.log("🔌 WebSocket ngắt kết nối:", event.code, event.reason);
      stopHeartbeat();
    };

    ws.onerror = (e) => {
      console.error("❌ Lỗi WebSocket:", e);
      stopHeartbeat();
    };
  }

  // ================== UI RENDER ==================
  function renderPlayers() {
    document.querySelector(".turn").innerHTML = `Lượt: ` + totalRounds;

    const container = document.getElementById("playersContainer");
    container.innerHTML = "";

    players.forEach((player, index) => {
      if (player.isActive) activeIndex = index;

      const card = document.createElement("div");
      const colorClass = index % 2 === 0 ? "red" : "white";
      card.className = `player-card ${colorClass}`;
      card.style.backgroundColor = colors[index % colors.length];
      if (player.isActive) card.classList.add("active-player");

      let extratimeDisplay = "";
      if (timeMode === 1) {
        if (player.extraTime === -1) {
          extratimeDisplay =
            '<i class="fas fa-infinity timeout-icon" style="color: #4CAF50;"></i> không giới hạn';
        } else if (player.extraTime === 0) {
          // ✅ SỬA: Khi hết lượt, hiển thị tất cả icon đồng hồ màu xám (không sáng)
          extratimeDisplay = Array.from({ length: timeOut })
            .map(() => {
              return '<i class="fas fa-stopwatch timeout-icon" style="color: #B0BEC5; margin: 0 2px;"></i>';
            })
            .join("");
        } else {
          // ✅ Icon đồng hồ còn lượt - màu xanh lá
          const remainingIcons = Array.from({ length: player.extraTime })
            .map(() => {
              return '<i class="fas fa-stopwatch timeout-icon" style="color: #4CAF50; margin: 0 2px;"></i>';
            })
            .join("");

          // ✅ Icon đồng hồ đã dùng - màu xám
          const usedIcons = Array.from({ length: timeOut - player.extraTime })
            .map(() => {
              return '<i class="fas fa-stopwatch timeout-icon" style="color: #B0BEC5; margin: 0 2px;"></i>';
            })
            .join("");

          extratimeDisplay = remainingIcons + usedIcons;
        }
      }

      card.innerHTML = `
    <div class="player-header">
      <input class="player-name" data-index="${index}" value="${
        player.name || ""
      }" />
      <div class="player-score" id="score${index}">${player.score}</div>
    </div>
    <div class="player-stats">
      <div class="stats-left">
        <div class="stat-item avg">AVG: ${Number(player.avg ?? 0).toFixed(
          3,
        )}</div>
        <div class="stat-item hr">HR: ${player.hr ?? 0}</div>
      </div>
      <div class="extratime" id="extratime${index}">${extratimeDisplay}</div>
    </div>
  `;
      container.appendChild(card);
    });

    document.querySelectorAll(".player-name").forEach((input) => {
      input.addEventListener("input", (e) => {
        const index = e.target.dataset.index;
        players[index].name = e.target.value;
        sendData(index, "change_name");
      });
    });
  }

  function updateUI() {
    renderPlayers();
  }



  // ================== GAME LOGIC ==================
  // trừ điểm
  btnDecreaseScore.addEventListener("click", () => {
    if (!gameStart || !gameRunning || (!isRunning && timeMode === 1)) return;
    btnDecreaseScore.disabled = true;
    btnDecreaseScore.style.opacity = "0.5";
    if (activeIndex >= 0) {
      players[activeIndex].score--;
      updateUI();
      sendData(activeIndex, "update_score");
    }
    setTimeout(() => {
      btnDecreaseScore.disabled = false;
      btnDecreaseScore.style.opacity = "1";
    }, lockOpacity);
  });
  // tăng điểm
  btnIncreaseScore.addEventListener("click", () => {
    if (!gameStart || !gameRunning || (!isRunning && timeMode === 1)) return;
    btnIncreaseScore.disabled = true;
    btnIncreaseScore.style.opacity = "0.5";
    if (activeIndex >= 0) {
      players[activeIndex].score++;
      updateUI();
      sendData(activeIndex, "update_score");
    }
    setTimeout(() => {
      btnIncreaseScore.disabled = false;
      btnIncreaseScore.style.opacity = "1";
    }, lockOpacity);
  });
  // xin thời gian
  btnMoreTime.addEventListener("click", () => {
    if (!gameStart || !gameRunning || (!isRunning && timeMode === 1)) return;
    showQuestionDetail();
  });
  // đổi lượt
  btnNextTurn.addEventListener("click", () => {
    if (!gameStart || !gameRunning || (!isRunning && timeMode === 1)) return;
    btnNextTurn.disabled = true;
    btnNextTurn.style.opacity = "0.5";
    nextTurn();
    setTimeout(() => {
      btnNextTurn.disabled = false;
      btnNextTurn.style.opacity = "1";
    }, lockOpacity);
  });

  document.querySelectorAll(".score-btn").forEach((btn) => {
    btn.addEventListener("click", () => {
      if (gameStart && gameRunning) {
        const value = parseInt(btn.dataset.score);
        if (activeIndex >= 0) {
          players[activeIndex].score += value;
          updateUI();
          sendData(activeIndex, "update_score");
        }
      }
    });
  });

  let canAskForTime = true;
  let countdownInterval = null;

  function showQuestionDetail() {
    if (!canAskForTime) return;
    if (players[activeIndex].extraTime === 0) return;
    if (players[activeIndex].extraTime > 0) {
      players[activeIndex].extraTime -= 1;
    }
    updateUI();
    sendData(activeIndex, "request_extra_time");

    canAskForTime = false;
    let countdown = 10;
    const originalText = btnMoreTime.textContent;

    btnMoreTime.textContent = `${countdown}s`;

    countdownInterval = setInterval(() => {
      countdown--;
      btnMoreTime.textContent = `${countdown}s`;

      if (countdown <= 0) {
        clearInterval(countdownInterval);
        btnMoreTime.textContent = originalText;
        canAskForTime = true;
      }
    }, 1000);
  }

  function nextTurn() {
    if (activeIndex < 0) return;

    players[activeIndex].isActive = false;
    activeIndex = (activeIndex + 1) % players.length;
    players[activeIndex].isActive = true;

    if (activeIndex === 0) {
      totalRounds++;
    }

    updateUI();
    stopCountdown();
    sendData(activeIndex, "switch_turn");
    renderPlayers();

    btnMoreTime.textContent = "XIN THỜI GIAN";
    canAskForTime = true;
    if (countdownInterval) {
      clearInterval(countdownInterval);
      countdownInterval = null;
    }
  }

  function updatePauseButton() {
    const btnPause = document.getElementById("btnPause");
    if (!btnPause) return;

    // Ẩn/hiện nút dựa trên timeMode
    if (timeMode === 1) {
      btnPause.classList.remove("btn-hidden");

      // Cập nhật trạng thái nút
      if (!isRunning) {
        btnPause.textContent = "TIẾP TỤC";
        btnPause.classList.remove("btn-yellow");
        btnPause.classList.add("btn-green");
      } else {
        btnPause.textContent = "TẠM DỪNG";
        btnPause.classList.remove("btn-green");
        btnPause.classList.add("btn-yellow");
      }
    } else {
      btnPause.classList.add("btn-hidden");
    }
    renderPlayers();
  }

  document.getElementById("btnPause").addEventListener("click", () => {
    if (!ws || ws.readyState !== WebSocket.OPEN) return;
    if (timeMode !== 1) return;
    if (gameStart && gameRunning) {
      isRunning = !isRunning;
      updatePauseButton();
      ws.send(
        JSON.stringify({
          type: "command",
          action: "toggle_pause",
          data: { isRunning: isRunning },
        }),
      );
    }
  });

  function disconnectAndRedirect() {
    sessionStorage.clear();
    window.location.href = "/index.html";
  }

  document.getElementById("btnDisconnect").addEventListener("click", () => {
    if (ws && ws.readyState === WebSocket.OPEN) {
      ws.send(JSON.stringify({ type: "disconnect" }));
      setTimeout(disconnectAndRedirect, 200);
    } else {
      disconnectAndRedirect();
    }
  });

  function sendData(activeIndex, action) {
    if (ws && ws.readyState === WebSocket.OPEN) {
      let payload = null;

      if (action === "switch_turn") {
        payload = {
          type: "command",
          action: "switch_turn",
          data: {
            playerId: players[activeIndex].id,
            isActive: players[activeIndex].isActive,
            turn: players[activeIndex].isActive,
          },
        };
      }

      if (action === "update_score") {
        payload = {
          type: "command",
          action: "update_score",
          data: {
            playerId: players[activeIndex].id,
            name: players[activeIndex].name,
            score: players[activeIndex].score,
            isActive: players[activeIndex].isActive,
          },
        };
      }

      if (action === "change_name") {
        payload = {
          type: "command",
          action: "change_name",
          data: {
            playerId: players[activeIndex].id,
            name: players[activeIndex].name,
          },
        };
      }

      if (action === "request_extra_time") {
        payload = {
          type: "command",
          action: "request_extra_time",
          data: {
            playerId: players[activeIndex].id,
            extraTime: players[activeIndex].extraTime,
            isActive: players[activeIndex].isActive,
          },
        };
      }

      if (payload) {
        ws.send(JSON.stringify(payload));
      }
    }
  }

  function startCountdown(endTime) {
    if (countdownInterval) clearInterval(countdownInterval);
    btnMoreTime.disabled = true;

    countdownInterval = setInterval(() => {
      const remaining = Math.max(0, Math.ceil((endTime - Date.now()) / 1000));
      btnMoreTime.textContent = `${remaining}s`;

      if (remaining <= 0) {
        clearInterval(countdownInterval);
        localStorage.removeItem("extraTimeEnd");
        btnMoreTime.disabled = false;
        btnMoreTime.textContent = "XIN THỜI GIAN";
      }
    }, 200);
  }

  function stopCountdown() {
    if (countdownInterval) {
      clearInterval(countdownInterval);
      countdownInterval = null;
    }

    btnMoreTime.disabled = false;
    btnMoreTime.textContent = "XIN THỜI GIAN";
    localStorage.removeItem("extraTimeEnd");
    canAskForTime = true;
  }

  window.addEventListener("load", () => {
    const savedEnd = localStorage.getItem("extraTimeEnd");
    if (savedEnd) {
      const endTime = Number(savedEnd);
      if (Date.now() < endTime) {
        startCountdown(endTime);
      } else {
        localStorage.removeItem("extraTimeEnd");
      }
    }
  });

  window.addEventListener("DOMContentLoaded", () => {
    connectWebSocket();
  });
})();
